/* * Author: Dubravko Penezic
 *
 * Version: 1.0, 2014
 *
 * DpCSSoilSensor.cpp - Library source code
 *
 * DpCSSoilSensor Arduino library provide function for gething
 * data form Capacitive Sensing Soil Sensor.
 *
 * Most part of code come from original DpCSSoilSensor library !
 *
 * Source code is provided as is, without any warranty.
 *
 * Distributetd under CC BY v 3.0
 *
 * http://pcb.daince.net/doku.php?id=arduino_code
 * */

#if ARDUINO >= 100
#include "Arduino.h"
#else
#include "WProgram.h"
#include "pins_arduino.h"
#include "WConstants.h"
#endif

#include "DpCSSoilSensor.h"

// Constructor /////////////////////////////////////////////////////////////////
// Function that handles the creation and setup of instances

DpCSSoilSensor::DpCSSoilSensor(uint8_t sendPin, uint8_t receivePin){

	uint8_t sPort, rPort;

	// initialize this instance's variables
	// Serial.begin(9600);		// for debugging
	error = 1;
	loopTimingFactor = 310;		// determined empirically -  a hack
	
	CS_Timeout_Millis = (2000 * (float)loopTimingFactor * (float)F_CPU) / 16000000;
    
	// get pin mapping and port for send Pin - from PinMode function in core

#ifdef NUM_DIGITAL_PINS
	if (sendPin >= NUM_DIGITAL_PINS) error = -1;
	if (receivePin >= NUM_DIGITAL_PINS) error = -1;
#endif
	
	sBit =  digitalPinToBitMask(sendPin);			// get send pin's ports and bitmask
	sPort = digitalPinToPort(sendPin);
	sReg = portModeRegister(sPort);
	sOut = portOutputRegister(sPort);				// get pointer to output register   

	rBit = digitalPinToBitMask(receivePin);			// get receive pin's ports and bitmask 
	rPort = digitalPinToPort(receivePin);
	rReg = portModeRegister(rPort);
	rIn  = portInputRegister(rPort);
   	rOut = portOutputRegister(rPort);
	
	// get pin mapping and port for receive Pin - from digital pin functions in Wiring.c
    noInterrupts();
	*sReg |= sBit;              // set sendpin to OUTPUT 
    interrupts();
}

// Public Methods //////////////////////////////////////////////////////////////
// Functions available in Wiring sketches, this library, and other libraries

float DpCSSoilSensor::getValue(){
	total = 0;
	if (samples == 0) return -10;         // no measuring
	if (error < 0) return -20;            // bad pin


	for (uint8_t i = 0; i < samples; i++) {    // loop for samples parameter - simple lowpass filter
		if (SenseOneCycle() < 0)  return -2;   // variable over timeout
	}

	if (total > CS_SS_Wet) return -41;                 // higher then wet         
	if (total < CS_SS_Dry) return -42;                 // lover then dry        

	return ((total-CS_SS_Dry)/(float)(CS_SS_Wet-CS_SS_Dry)*100);
}


long DpCSSoilSensor::getRaw(){
	total = 0;
	if (samples == 0) return 0;
	if (error < 0) return -1;                  // bad pin - this appears not to work

	for (uint8_t i = 0; i < samples; i++) {    // loop for samples parameter - simple lowpass filter
		if (SenseOneCycle() < 0)  return -2;   // variable over timeout
	}

	return total;
}


void DpCSSoilSensor::setSamples(uint8_t samp){
	samples = samp;
}

uint8_t DpCSSoilSensor::getSamples(){
  return samples;
}

void DpCSSoilSensor::setTimeoutMillis(unsigned long timeoutm){
        timeoutMillis = timeoutm;
	CS_Timeout_Millis = (timeoutMillis * (float)loopTimingFactor * (float)F_CPU) / 16000000;  // floats to deal with large numbers
}

unsigned long DpCSSoilSensor::getTimeoutMillis() {
	return timeoutMillis;
}

void DpCSSoilSensor::setDry(long CSValue){
	CS_SS_Dry = CSValue;
}

long DpCSSoilSensor::getDry(){
	return CS_SS_Dry;
}

void DpCSSoilSensor::setWet(long CSValue){
	CS_SS_Wet = CSValue;
}

long DpCSSoilSensor::getWet(){
	return CS_SS_Wet;
}



// Private Methods /////////////////////////////////////////////////////////////
// Functions only available to other functions in this library

int DpCSSoilSensor::SenseOneCycle(void)
{
    noInterrupts();
	*sOut &= ~sBit;        // set Send Pin Register low
	
	*rReg &= ~rBit;        // set receivePin to input
	*rOut &= ~rBit;        // set receivePin Register low to make sure pullups are off
	
	*rReg |= rBit;         // set pin to OUTPUT - pin is now LOW AND OUTPUT
	*rReg &= ~rBit;        // set pin to INPUT 

	*sOut |= sBit;         // set send Pin High
    interrupts();

	while ( !(*rIn & rBit)  && (total < CS_Timeout_Millis) ) {  // while receive pin is LOW AND total is positive value
		total++;
	}
    
	if (total > CS_Timeout_Millis) {
		return -2;         //  total variable over timeout
	}
   
// set receive pin HIGH briefly to charge up fully - because the while loop above will exit when pin is ~ 2.5V 
    noInterrupts();
	*rOut  |= rBit;        // set receive pin HIGH - turns on pullup 
	*rReg |= rBit;         // set pin to OUTPUT - pin is now HIGH AND OUTPUT
	*rReg &= ~rBit;        // set pin to INPUT 
	*rOut  &= ~rBit;       // turn off pullup

	*sOut &= ~sBit;        // set send Pin LOW
    interrupts();

	while ( (*rIn & rBit) && (total < CS_Timeout_Millis) ) {  // while receive pin is HIGH  AND total is less than timeout
		total++;
	}
	// Serial.println(total);

	if (total >= CS_Timeout_Millis) {
		return -2;     // total variable over timeout
	} else {
		return 1;
	}
}
